package rs.ac.bg.fon.ai.inteligentniSistemi.sparql.query;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.jena.query.QueryExecution;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.query.QuerySolution;
import org.apache.jena.query.ResultSet;
import org.apache.jena.rdf.model.Literal;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.jena.rdf.model.Resource;

public class QueryExecutor {

	/**
	 * Executes SPARQL SELECT query over a given model and returns a list of values
	 * for the given variable name. This method should be used when only one variable 
	 * values are retrieved from a SELECT SPARQL query.
	 * 
	 * @param query
	 * @param variable
	 * @param graph
	 * @return List of values for the specified variable
	 */
	public List<String> executeSelectQueryOverModel(String query,
			String variable, Model graph) {
		
		// Execute the query over the model
		QueryExecution qe = QueryExecutionFactory.create(query, graph);
		return executeSelectQuery(variable, qe);
	}
	
	/**
	 * Executes SPARQL SELECT query over a specified QueryExecution instance and 
	 * returns a list of values for the given variable name. This method should be used when only one variable 
	 * values are retrieved from a SELECT SPARQL query.
	 * 
	 * @param query
	 * @param variable
	 * @param qe
	 * @return List of values for the specified variable
	 */
	private List<String> executeSelectQuery(String variable, QueryExecution qe) {
		// Execute the query over the model
		ResultSet resultSet = qe.execSelect();
		
		List<String> results = new LinkedList<String>();
		
		// obtain results from the result set
		while (resultSet.hasNext()) {
			QuerySolution solution = resultSet.nextSolution();
			RDFNode value = solution.get(variable);
			
			if (value.isLiteral())
				results.add(((Literal) value).getLexicalForm());
			else
				results.add(((Resource) value).getURI());
		}
		qe.close();
		
		return results;
	}
	
	/**
	 * Executes SPARQL SELECT query over a given model and returns a list of values
	 * for all variable names.
	 * 
	 * @param query
	 * @param graph
	 * @return List of arrays of variable values. Each list element corresponds to a 
	 * record from the result set
	 */
	public List<Object[]> executeSelectQueryOverModel(String query, Model graph) {
		
		// Execute the query over the model
		QueryExecution qe = QueryExecutionFactory.create(query, graph);
		return executeSelectQuery(qe);
	}
	
	private List<Object[]> executeSelectQuery(QueryExecution qe) {
		ArrayList<Object[]> records = new ArrayList<Object[]>();
		
		ResultSet resultSet = qe.execSelect();
		
		while (resultSet.hasNext()) {
			Object[] varValues = new Object[resultSet.getResultVars().size()];
			
			QuerySolution result = (QuerySolution) resultSet.next();
			
			Iterator<String> variables = result.varNames();
			
			int varIndex = 0;
			
			while (variables.hasNext()) {
				String var = (String) variables.next();
				
				RDFNode value = result.get(var);
				
				if (value.isLiteral())
					varValues[varIndex++] = ((Literal) value).getLexicalForm();
				else
					varValues[varIndex++] = ((Resource) value).getURI();
			}
			records.add(varValues);
		}
		
		return records;
	}
	
	/**
	 * Executes SPARQL DESCRIBE query over a specified model and returns a Model instance 
	 * with triples retrieved as a result.
	 * 
	 * @param query
	 * @param model
	 * @return Model containing a subgraph of the original model containing all sentences 
	 * where query result(s) is a subject.
	 */
	public Model executeDescribeSparqlQuery(String query, Model model) {
		// Execute the query and obtain results
		QueryExecution qe = QueryExecutionFactory.create(query, model);
		Model resultModel = qe.execDescribe();
		
		// Important - free up resources used running the query
		qe.close();
		
		return resultModel;
	}
	
	/**
	 * Executes SPARQL SELECT query over a specified SPARQL endpoint and returns a list of values
	 * for the given variable name. This method can only be applied if SELECT query retrieves 
	 * one variable value.
	 * 
	 * @param query
	 * @param variable
	 * @param sparqlEndpoint
	 * @return
	 */
	public List<String> executeSelectQueryOverSparqlEndpoint(String query,
			String variable, String sparqlEndpoint) {
		
		// Execute the query over the model
		QueryExecution qe = QueryExecutionFactory.sparqlService(sparqlEndpoint, query);
		return executeSelectQuery(variable, qe);
	}
	
	/**
	 * Executes SPARQL SELECT query over a specified SPARQL endpoint and returns a list of values
	 * for all variable names.
	 * 
	 * @param query
	 * @param sparqlEndpoint
	 * @return
	 */
	public List<Object[]> executeSelectSparqlQuery(String query, String sparqlEndpoint) {
		QueryExecution qe = QueryExecutionFactory.sparqlService(sparqlEndpoint, query);
		return executeSelectQuery(qe);
	}

}
