package rs.ac.bg.fon.ai.inteligentniSistemi.sparql.services;

import java.util.List;

import org.apache.jena.rdf.model.Model;

import rs.ac.bg.fon.ai.inteligentniSistemi.sparql.query.QueryExecutor;
import rs.ac.bg.fon.ai.inteligentniSistemi.sparql.util.Constants;

public class MovieService {
	
	private Model model;
	private QueryExecutor queryExecutor = new QueryExecutor();
	
	public MovieService (Model model) {
		this.model = model;
	}
	
	/**
	 * Retrieves all classes in the graph.
	 * 
	 * @return list of names
	 */
	public List<String> getAllClasses() {
		String query = 
			"PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> " + 
			"SELECT DISTINCT ?clazz " +
			"WHERE { " +
				"?s rdf:type ?clazz . " +
			"}";
		
		return queryExecutor
				.executeSelectQueryOverModel(query, "clazz", model);
	}
	
	/**
	 * Retrieves all instances of the class Person.
	 *  
	 * @return list of URIs
	 */
	public List<String> getAllPersons() {
		String query = 
				"PREFIX rdf: <"+Constants.RDF+">" +
				"PREFIX schema: <"+Constants.SCHEMA+">" +
				"SELECT ?person " +
				"WHERE {" +
					"?person rdf:type schema:Person . " +
				"}";
		
		return queryExecutor.executeSelectQueryOverModel(query, "person", model);
	}
	
	/**
	 * Describes resource with given URI.
	 * 
	 * @param uri
	 * @return model containing triples with given URI as a subject
	 */
	public Model describeResource(String uri) {
		String query = 
			"DESCRIBE <" + uri + ">";
		
		return queryExecutor	.executeDescribeSparqlQuery(query, model);
	}
	
	/**
	 * Retrieves all instances of the class Movie.
	 *  
	 * @return list of URIs
	 */
	public List<String> getAllMovies() {
		String query = 
			"PREFIX rdf: <"+Constants.RDF+">" +
			"PREFIX schema: <"+Constants.SCHEMA+">" +
			"SELECT ?movie " +
			"WHERE {" +
				"?movie rdf:type schema:Movie . " +
			"}";
		
		return queryExecutor.executeSelectQueryOverModel(query, "movie", model);
	}
	
	
	/**
	 * Retrieves names of all persons in the graph.
	 * 
	 * @return list of names
	 */
	public List<String> getAllPersonNames () {
		String query = 
			"PREFIX schema: <"+Constants.SCHEMA+"> " + 
			"PREFIX rdf: <"+Constants.RDF+"> " + 
			"SELECT ?name " +
			"WHERE { " +
				"?person rdf:type schema:Person . " +
				"?person schema:name ?name . " +
			"}";
		
		return queryExecutor
				.executeSelectQueryOverModel(query, "name", model);
	}
	
	/**
	 * Retrieves names of all movies that are released after the date 
	 * 01.01.2010.
	 * 
	 * @return list of movie names
	 */
	public List<String> getAllMoviesNewerThan2010() {
		String queryString = 
			"PREFIX schema: <"+Constants.SCHEMA+"> " + 
			"PREFIX rdf: <"+Constants.RDF+"> " + 
			"SELECT ?name " + 
			"WHERE  { " + 
				"?movie rdf:type schema:Movie ." +
				"?movie schema:name ?name ." + 
				"?movie schema:datePublished ?date ." + 
				
				"FILTER (?date > \"2010-01-01\")" +
			"}";

		return queryExecutor
				.executeSelectQueryOverModel(queryString, "name", model);
	}
	
	/**
	 * Retrieves names of all movies that are directed by J. J. Abrams.
	 * 
	 * @return list of movie names
	 */
	public List<String> getAllMoviesByJJAbrams() {
		String query = 
			"PREFIX schema: <"+Constants.SCHEMA+"> " + 
			"PREFIX rdf: <"+Constants.RDF+"> " + 
			"SELECT DISTINCT ?name " +
			"WHERE { " +
				"?movie rdf:type schema:Movie . " +
				"?movie schema:name ?name . " +
				"?movie schema:director ?director . " +

				"?director rdf:type schema:Person . " +
				"?director schema:name \"J.J. Abrams\" . " +
			"}";

		return queryExecutor
				.executeSelectQueryOverModel(query, "name", model);
	}
}
