package rs.ac.bg.fon.ai.sparqllab;


import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.io.Reader;
import java.util.Iterator;

import org.apache.jena.query.QueryExecution;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.query.QuerySolution;
import org.apache.jena.query.ResultSet;
import org.apache.jena.query.ResultSetFactory;
import org.apache.jena.query.ResultSetFormatter;
import org.apache.jena.query.ResultSetRewindable;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.jena.sparql.engine.http.QueryEngineHTTP;


public class SPARQLUtil {
	
	public static final String DBPEDIA_SPARQL_ENDPOINT = "http://dbpedia.org/sparql";
	
	public static ResultSetRewindable executeSelectQueryOverSparqlEndpoint(String query, String sparqlEndpoint) {
		QueryExecution qe = QueryExecutionFactory.sparqlService(sparqlEndpoint, query);
		
		ResultSet results = ResultSetFactory.copyResults(qe.execSelect());
		
		// Important - free up resources used running the query
		qe.close();
		
		return ResultSetFactory.makeRewindable(results);
	}
	
	public static ResultSetRewindable executeSelectQueryOverModel(String query, Model model) {
		QueryExecution qe = QueryExecutionFactory.create(query, model);
		
		ResultSet results = ResultSetFactory.copyResults(qe.execSelect());
		
		qe.close();
		
		return ResultSetFactory.makeRewindable(results);
	}
	
	public static boolean executeAskSparqlQuery(String query, String sparqlEndpoint) {
		QueryExecution qe = QueryExecutionFactory.sparqlService(sparqlEndpoint, query);
		boolean result = qe.execAsk();
		
		qe.close();
		
		return result;
	}
	
	public static Model executeDescribeSparqlQuery(String query, String sparqlEndpoint) {
		
		QueryExecution qe = QueryExecutionFactory.sparqlService(sparqlEndpoint, query);
		Model resultModel = qe.execDescribe();
		
		qe.close();
		
		return resultModel;
	}
	
	public static Model executeConstructSparqlQuery(String query, String sparqlEndpoint) {
		
		QueryExecution qe = QueryExecutionFactory.sparqlService(sparqlEndpoint, query);
		
		// the following 2 lines are introduced to prevent the occurrence of exception caused by 
		// incompatible characters in the result set
		// http://stackoverflow.com/questions/31508580/riotexception-in-execconstruc-method-queryexecution-class-using-jena-2-13
		QueryEngineHTTP qeHttp = (QueryEngineHTTP) qe;
		qeHttp.setModelContentType("application/rdf+xml");
		
		Model resultModel = qeHttp.execConstruct();
		
		qe.close();
		
		return resultModel;
	}
	
	
	public static void printSPARQLSelectResults(ResultSet resultSet, PrintStream out) {
		
		while (resultSet.hasNext()) {
			QuerySolution result = (QuerySolution) resultSet.next();
			
			Iterator<String> variables = result.varNames();

			while (variables.hasNext()) {
				String var = (String) variables.next();

				RDFNode value = result.get(var);

				if (value.isLiteral())
					out.println(var + ": " + value.asLiteral().getLexicalForm());
				else if ( value.isURIResource() )
						out.println(var + ": " + value.asResource().getURI());
				else out.println(var + ": BNode: " + value.asNode().getBlankNodeLabel());

			}
			out.println("----------------------------");
		}
	}
	
	public static void serializeResultsToFile(String fileName, ResultSet results, String format) {
			
		try {
			FileOutputStream out = new FileOutputStream(fileName);
			if ( format.equalsIgnoreCase("JSON") )
				ResultSetFormatter.outputAsJSON(out, results);
			else
				ResultSetFormatter.outputAsXML(out, results);
			out.close();
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
	}
	
	public static void serializeResultsToFile(String fileName, Model results, String format) {
		
		try {
			FileOutputStream out = new FileOutputStream(fileName);
			results.write(out, format);
			out.close();
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
	}
	
	public static int getResultsCount(ResultSet resultSet) {
		int cnt = 0;
		while (resultSet.hasNext()) {
			resultSet.next();
			cnt++;
		}
		return cnt;
	}
	
	public static String readTextFromFile(String fileName, String encoding) throws IOException {
		
		StringBuilder sb = new StringBuilder();
		
		Reader reader = new InputStreamReader(new FileInputStream(fileName), encoding);
		BufferedReader in = new BufferedReader(reader);
		
		String line = null;
		while ( (line = in.readLine()) != null ) {
			sb.append( line );
			sb.append("\n");
		}
		in.close();
		
		return sb.toString();
		
	}

}
