from pathlib import Path
import json
import csv
import functools

## Task 2
def log_execution(func):
    @functools.wraps(func)
    def wrapper_log_execution(*args, **kwargs):
        from datetime import datetime
        import time

        started = datetime.strftime(datetime.now(), "%Y-%m-%d %H:%M:%S")
        start_time = time.perf_counter()

        value = func(*args, **kwargs)

        exec_time = (time.perf_counter() - start_time) * 1000

        args_str = ",".join([str(arg) for arg in args])
        kwargs_str = ",".join([f"{key}={val}" for key, val in kwargs.items()])
        func_name = f"{func.__name__}({args_str if args_str != '' else 'NO_args'}, {kwargs_str if kwargs_str != '' else 'NO_kwargs'})"

        with open('execution_log.csv', 'a') as fcsv:
            csv_writer = csv.writer(fcsv, delimiter = ';')
            csv_writer.writerow((started, func_name, exec_time))

        return value

    return wrapper_log_execution


def load_data_from_json(fname):
    from sys import stderr
    try:
        with open(fname, "r") as fobj:
            return json.load(fobj)
    except IOError as err:
        stderr.write(f"An error occurred when trying to read from file {fname}:\n{err}")


## Task 1.1
def get_completed_per_user(todos):
    from collections import defaultdict

    completed_counts = defaultdict(int)
    for todo_item in todos:
        if todo_item['completed']:
            completed_counts[todo_item['userId']] += 1
    return completed_counts


@log_execution
def users_with_most_completed_tasks(todos):
    completed_counts = get_completed_per_user(todos)

    max_completed = max(completed_counts.values())
    top_users = [user for user, complete_cnt in completed_counts.items() if complete_cnt == max_completed]
    print("Users with max completed items: " + ", ".join([str(user) for user in top_users]))


## Task 1.2
@log_execution
def add_task_priority(todos, out_fname):
    from random import randint, seed

    seed(7)
    for item in todos:
        item['priority'] = randint(1,5)

    with open(out_fname, "w") as fobj:
        json.dump(todos, fobj, indent=4)


## Task 1.3
@log_execution
def prepare_user_assignments(todos):
    from collections import defaultdict

    def get_assignments_dir():
        assign_dir = Path.cwd() / 'user_assignments'
        if not assign_dir.exists(): assign_dir.mkdir()
        return assign_dir

    user_assignments = defaultdict(list)
    for item in todos:
        if not item['completed']:
            user_assignments[item['userId']].append((item['id'], item['priority'], item['title']))

    for user_id, assignments in user_assignments.items():
        with open(get_assignments_dir() / str(user_id), 'w') as fobj:
            csv_writer = csv.writer(fobj)
            csv_writer.writerow(("id", "priority", "title"))
            for item in sorted(assignments, key=lambda item: item[1]):
                csv_writer.writerow(item)


if __name__ == '__main__':

    # učitavanje podataka iz json fajla u listu rečnika
    tasks = load_data_from_json("todos.json")

    # prikaz članova tima sa najvećim brojem završenih zadataka
    users_with_most_completed_tasks(tasks)

    # nasumična dodela prioriteta zadacima i upisivanje tako ažuriranih podataka u novi json fajl
    add_task_priority(tasks, "prioritised_todos.json")

    # kreiranje liste zadataka, sortirane po prioritetu zadataka, za svakog člana tima i njen upis u csv fajl
    prepare_user_assignments(tasks)